(*^ This file describes `cells` - a kind of typed rtti pointers.
    It allows to read an write into the memory via safe interface.

    Roughly, the cells are structured as trees.
    The children are different subcomponents, that have separate meaning.
    For example, the fields of a struct, or the elements of an array.
    
    Beware, it is not specified, what children will be present and the level of nesting.
    For example, it is not specified, whether each axis of an array gets its own cell, or not.
*)

TYPE
    (**
        Typed pointer to some data. See top-level documentation for details.

        Unless said otherwise, all functions assume and ensure `RTTI_CELL` is valid.
    *)
    RTTI_CELL: STRUCT
        ty: RTTI_TYPE;                         (*< type of the variable *)
        addr: REF_TO VOID;                     (*< addr of the variable *)
        root: REF_TO RTTI_ROOT_VAR_DATA;       (*< pointer to root variable data *)
        pos: RTTI_POS;                         (*< total index in the root type *)
    END_STRUCT
END_TYPE

(** check if this cell is `valid`, or not.

    `invalid` cells represent a kinda of a NULL value.
    They appear when an expected failure happens.
    For example, if `rtti_root_var_find_cell_by_pos` did not find the cell, it returns an `invalid` one.
    
    Usually, functions return `valid` rtti_cells, unless said otherwise.
*)
FUNCTION rtti_cell_is_valid: BOOL
VAR_INPUT v: RTTI_CELL; END_VAR
VAR nil: RTTI_TYPE; END_VAR
    rtti_cell_is_valid := v.ty <> nil;
END_FUNCTION

(** returns the position of current cell
    can be used in `rtti_root_var_find_cell_by_pos` to get this cell back

    TODO: unowned cells, may change this behaviour!
*)
FUNCTION rtti_cell_pos: RTTI_POS
VAR_INPUT v: RTTI_CELL; END_VAR
    rtti_cell_pos := v.pos;
END_FUNCTION

(** returns the type of the value this cell points to *)
FUNCTION rtti_cell_type: RTTI_TYPE
VAR_INPUT v: RTTI_CELL; END_VAR
    rtti_cell_type := v.ty;
END_FUNCTION

(** returns the parent cell; for root cell returns `invalid` *)
FUNCTION rtti_cell_parent: RTTI_CELL
VAR_INPUT v: RTTI_CELL; END_VAR
    if v.pos <> 0 then
        rtti_cell_parent := __rtti_cell_parent_for_pos(__rtti_root_of(v), v.pos);
    end_if
END_FUNCTION

(** returns the next sibling cell, or the next sibling of the parent.
    returns invalid, if this is the last (at any level) cell in the whole root cell.

    TODO: this function is rather vague, and should probably be replaced with a few specialized variants.
    Use `rtti_cell_nth_child` instead.
*)
FUNCTION rtti_cell_next: RTTI_CELL
VAR_INPUT v: RTTI_CELL; END_VAR
    rtti_cell_next := __rtti_cell_by_pos(__rtti_root_of(v), v.pos + __rtti_type_total_len(v.ty));
END_FUNCTION

(** returns the previous sibling cell, or the previous sibling of the parent.
    returns `invalid`, if this is the first (at any level) cell in the whole root cell.
    
    TODO: this function is rather vague, and should probably be replaced with a few specialized variants.
    Use `rtti_cell_nth_child` instead.
*)
FUNCTION rtti_cell_prev: RTTI_CELL
VAR_INPUT v: RTTI_CELL; END_VAR
VAR prev: RTTI_CELL; END_VAR
    prev := __rtti_cell_parent_for_pos(__rtti_root_of(v), v.pos - 1);
    // check if we are the first field
    if prev.pos + 1 <> v.pos then
        rtti_cell_prev := prev;
    end_if
END_FUNCTION

(** returns the number of children cells.
    If = 0, the current cell does not have children
*)
FUNCTION rtti_cell_n_children: USIZE
VAR_INPUT v: RTTI_CELL; END_VAR
    rtti_cell_n_children := __rtti_type_nfields(v.ty);
END_FUNCTION

(** gets a child cell by its index (among the immediate children of this cell).

    assumes `ix < nchildren`. Otherwise, does one of the following:
    1. Results in UB
    2. Panics
    3. Returns an `invalid` cell.
*)
FUNCTION rtti_cell_nth_child: RTTI_CELL
VAR_INPUT v: RTTI_CELL; END_VAR
VAR_INPUT ix: USIZE; END_VAR
VAR
    ty: RTTI_TYPE;
    field: __RTTI_TYPE_FLD;
END_VAR
VAR nil_fld: __RTTI_TYPE_FLD; END_VAR
    ty := rtti_cell_type(v);
    field := __rtti_type_field(ty, ix);
    if field = nil_fld then
        return;
    end_if

    rtti_cell_nth_child := __rtti_cell_field(v, field);
END_FUNCTION

(** Get thi child name or description. Intended for display use.

    See remarks about `rtti_type_display_name`!
*)
FUNCTION rtti_cell_nth_child_display_name: STRING
VAR_INPUT v: RTTI_CELL; END_VAR
VAR_INPUT ix: USIZE; END_VAR
VAR
    ty: RTTI_TYPE;
    field: __RTTI_TYPE_FLD;
END_VAR
VAR nil_fld: __RTTI_TYPE_FLD; END_VAR
    ty := rtti_cell_type(v);
    field := __rtti_type_field(ty, ix);
    if field = nil_fld then
        return;
    end_if

    __string_copy(field^.name, rtti_cell_nth_child_display_name);
END_FUNCTION

(* Get thie child name or description AS POINTER TO CHAR. Intended for display use.

    See remarks about `rtti_type_display_name`!
*)

FUNCTION rtti_cell_nth_child_display_pname: REF_TO CHAR
VAR_INPUT v: RTTI_CELL; END_VAR
VAR_INPUT ix: USIZE; END_VAR
VAR
    ty: RTTI_TYPE;
    field: __RTTI_TYPE_FLD;
END_VAR
VAR nil_fld: __RTTI_TYPE_FLD; END_VAR
    ty := rtti_cell_type(v);
    field := __rtti_type_field(ty, ix);
    if field = nil_fld then
        return;
    end_if

    rtti_cell_nth_child_display_pname:=field^.name;
END_FUNCTION

FUNCTION __rtti_cell_field: RTTI_CELL
VAR_INPUT
    v: RTTI_CELL;
    field: __RTTI_TYPE_FLD;
END_VAR
    __rtti_cell_field := (
        addr := v.addr + field^.offset,
        ty := field^.ty,
        root := v.root,
        pos := v.pos + field^.relative_pos,
    );
END_FUNCTION

FUNCTION __rtti_root_of: RTTI_CELL
VAR_INPUT v: RTTI_CELL; END_VAR
    __rtti_root_of := rtti_root_var_to_cell(__rtti_cell_to_root(v));
END_FUNCTION

FUNCTION __rtti_cell_to_root: RTTI_ROOT_VAR
VAR_INPUT v: RTTI_CELL; END_VAR
    __rtti_cell_to_root.data := v.root;
END_FUNCTION


(** reads the value from the cell to the buffer. Returns an error code.
    0 - means success, negative values mean error.

    The buffer content should be interpreted according to the rtti_type_kind.
    NB. RTTI_OPAQUE means the content of the buffer should not be interpreted ever, only copied around.

    asserts `sz = sizeof(Type)`. Returns an error otherwise.
    assumes the caller has synchronized access to the memory with other threads.
*)
FUNCTION rtti_cell_read: ISIZE
VAR_INPUT
    v: RTTI_CELL;
    buf: REF_TO VOID;
    sz: USIZE;
END_VAR
VAR
    ty_size: USIZE;
END_VAR
    ty_size := rtti_type_size(rtti_cell_type(v));
    if ty_size <> sz then
        rtti_cell_read := -1;
        return;
    end_if
	if buf=0 then
        rtti_cell_read := -2;
        return;
    end_if
    memcpy(buf, v.addr, ty_size);
    rtti_cell_read := DINT#ty_size;
END_FUNCTION

(** writes the value to the cell from the buffer. Returns an error code.
    0 - means success, negative values mean error.

    The buffer content should be interpreted according to the rtti_type_kind.
    NB. RTTI_OPAQUE means the content of the buffer should not be interpreted ever, only copied around.

    asserts `sz = sizeof(Type)`. Returns an error otherwise.
    assumes the caller has synchronized access to the memory with other threads.
*)
FUNCTION rtti_cell_write: ISIZE
VAR_INPUT
    v: RTTI_CELL;
    buf: REF_TO VOID;
    sz: USIZE;
END_VAR
VAR
    ty_size: USIZE;
END_VAR
    ty_size := rtti_type_size(rtti_cell_type(v));
    if ty_size <> sz then
        rtti_cell_write := -1;
        return;
    end_if

	if buf=0 then
		rtti_cell_write := -2;
		return;
    end_if
    memcpy(v.addr, buf, ty_size);
    rtti_cell_write := ISIZE#ty_size;

END_FUNCTION
