(** The RTTI_MODULE represents the core RTTI datastructure.
    It allows access to variables in the current compilation artifact (e.g. exe, or dylib).

    RTTI_MODULE is safe to send to another dylib, possibly having a separate copy of rtti functions,
    as long as they have exactly the same version of the rtti library.
*)

VAR_GLOBAL
    __CASTLE_RTTI_MODULE: RTTI_MODULE_DATA; // compiler magic
END_VAR

TYPE
    (** The RTTI_MODULE represents the core RTTI datastructure.
    
        Unless said otherwise, assumes and ensures RTTI_MODULE is valid
    *)
    RTTI_MODULE: REF_TO RTTI_MODULE_DATA;
    RTTI_MODULE_DATA: STRUCT
        first: REF_TO RTTI_ROOT_VAR_DATA;
        by_id: RTTI_HASHTABLE_BY_ID;
    END_STRUCT
    RTTI_HASHTABLE_BY_ID: STRUCT
        data: REF_TO REF_TO RTTI_ROOT_VAR_DATA;
        len: USIZE;
    END_STRUCT
END_TYPE

(** Get module for this compilation artifact *)
FUNCTION rtti_get_local_module: RTTI_MODULE
    rtti_get_local_module := ADR(__CASTLE_RTTI_MODULE);
END_FUNCTION

(** get the first variable in this module.
    Variables are organized in a linked list[^1], and this method allows traversing the variables.

    [^1]: only the interface; implementation may vary. 
*)
FUNCTION rtti_module_first_var: RTTI_ROOT_VAR
VAR_INPUT module: RTTI_MODULE; END_VAR
    rtti_module_first_var.data := module^.first;
END_FUNCTION

(** find a root variable by its id or returns an `invalid` variable

    NB. check [`rtti_root_var_is_valid`]
*)
FUNCTION rtti_module_var_by_id: RTTI_ROOT_VAR
VAR_INPUT 
    module: RTTI_MODULE;
    id: RTTI_ID;
END_VAR
VAR
    idx, len: USIZE;
    v: REF_TO RTTI_ROOT_VAR_DATA;
    ptr: REF_TO REF_TO RTTI_ROOT_VAR_DATA;
END_VAR
VAR nil_var: REF_TO RTTI_ROOT_VAR_DATA; END_VAR
    len := module^.by_id.len;
    idx := USIZE#id MOD len;
    ptr := module^.by_id.data;

    repeat
        v := (ptr + idx)^;
        idx := (idx + 1) MOD len;
        if v = nil_var then
            // Nothing found, exit with invalid var
            exit;
        end_if
    until v^.id = id
    end_repeat

    rtti_module_var_by_id.data := v;
END_FUNCTION


