(*^
    Root variables. They represent a typed static place in the binary.
    Root variables cannot be nested.

    Root variables allow for top-level traversal and options.
    To actually read, write, or access subfields, you need to obtain an `RTTI_CELL` from this variable.
    See `rtti_root_var_to_var`.

*)

(* RTTI_ROOT_VAR_DATA flags *)
VAR_GLOBAL CONSTANT
    RTTI_PERSISTENT: BYTE := 16#01;
    RTTI_RETAIN: BYTE := 16#02;
END_VAR

TYPE
    (**
        Root variables. They represent a typed static place in the binary.
        Root variables cannot be nested.

        See top-level documentation.

        Unless said otherwise, all functions assume and ensure `rtti_root_var` is valid.
    *)
    RTTI_ROOT_VAR: STRUCT
        data: REF_TO RTTI_ROOT_VAR_DATA;
    END_STRUCT
    RTTI_ROOT_VAR_DATA: STRUCT
        id: RTTI_ID;                    (* shallow id of the root variable *)
        ty: RTTI_TYPE;                  (* type of the variable *)
        name: REF_TO CHAR;
        addr: REF_TO VOID;              (* address of the value *)

        next: REF_TO RTTI_ROOT_VAR_DATA;
        prev: REF_TO RTTI_ROOT_VAR_DATA;

        flags: BYTE;                    (* bitflags, see var global constant sections above *)
    END_STRUCT
END_TYPE

(** check if this var is `valid`, or not.

    `invalid` vars represent a kinda of a NULL value.
    They appear when an expected failure happens.
    For example, if `rtti_module_var_by_id` did not find the variable, it returns an `invalid` one.
    
    Usually, functions return `valid` variables, unless said otherwise.
*)
FUNCTION rtti_root_var_is_valid: BOOL
VAR_INPUT v: RTTI_ROOT_VAR; END_VAR
VAR nil: REF_TO RTTI_ROOT_VAR_DATA; END_VAR
    rtti_root_var_is_valid := v.data <> nil;
END_FUNCTION

(** Get the next root variable in this module.
    Returns whether there was a next variable or not.
    Mutates the variable inplace, if there was a next variable.

    ```
    // iterate over all root variables
    v := rtti_module_first_var(module);
    repeat
        // process variable v
    unless not rtti_root_var_next(v)
    end_repeat
    ```
 *)
FUNCTION rtti_root_var_next: BOOL
VAR_IN_OUT v: RTTI_ROOT_VAR; END_VAR
VAR data: REF_TO RTTI_ROOT_VAR_DATA; END_VAR
VAR nil_data: REF_TO RTTI_ROOT_VAR_DATA; END_VAR
    data := v.data^.next;
    if data <> nil_data then
        v.data := data;
        rtti_root_var_next := TRUE;
    end_if
END_FUNCTION

(** Get the previous root variable in this module.
    Returns whether there was a previous variable or not.
    Mutates the variable inplace, if there was a previous variable.

    Oposite of `rtti_root_var_next`, if that returned true.
 *)
FUNCTION rtti_root_var_prev: BOOL
VAR_IN_OUT v: RTTI_ROOT_VAR; END_VAR
VAR data: REF_TO RTTI_ROOT_VAR_DATA; END_VAR
VAR nil_data: REF_TO RTTI_ROOT_VAR_DATA; END_VAR
    data := v.data^.prev;
    if data <> nil_data then
        v.data := data;
        rtti_root_var_prev := TRUE;
    end_if
END_FUNCTION

(** Convert the corresponding root var to a cell to its content *)
FUNCTION rtti_root_var_to_cell: RTTI_CELL
VAR_INPUT v: RTTI_ROOT_VAR; END_VAR
VAR data: REF_TO RTTI_ROOT_VAR_DATA; END_VAR
    rtti_root_var_to_cell.addr := v.data^.addr;
    rtti_root_var_to_cell.ty := v.data^.ty;
    rtti_root_var_to_cell.root := v.data;
END_FUNCTION

(** Returns the amount of positions this root var occupised

    It is guaranteed, that all descendants cell positions will be in the range:
    `0 <= pos < pos_size`.
*)
FUNCTION rtti_root_var_pos_size: RTTI_POS
VAR_INPUT v: RTTI_ROOT_VAR; END_VAR
    rtti_root_var_pos_size := __rtti_type_total_len(v.data^.ty);
END_FUNCTION

(** Find a cell, by its position.
    Returns an `invalid` cell, if there is no cell with corresponding position.

    checks `0 <= pos < pos_size`. Otherwise returns `invalid` cell.

    [`position`]: `rtti_cell_pos`

*)
FUNCTION rtti_root_var_find_cell_by_pos: RTTI_CELL
VAR_INPUT
    v: RTTI_ROOT_VAR;
    pos: RTTI_POS;
END_VAR
VAR data: REF_TO RTTI_ROOT_VAR_DATA; END_VAR
    rtti_root_var_find_cell_by_pos := __rtti_cell_by_pos(rtti_root_var_to_cell(v), pos);
END_FUNCTION

FUNCTION rtti_root_var_id: RTTI_ID
VAR_INPUT v: RTTI_ROOT_VAR; END_VAR
    rtti_root_var_id := v.data^.id;
END_FUNCTION

(** qualifed name of this root variable

    This can be just plain `my_var` for globals,
    and a qualifed `my_prg.my_var` for variables in programs.
*)
FUNCTION rtti_root_var_qname: STRING
VAR_INPUT v: RTTI_ROOT_VAR; END_VAR
    __string_copy(v.data^.name, rtti_root_var_qname);
END_FUNCTION

FUNCTION rtti_root_var_is_retain : BOOL
VAR_INPUT v: RTTI_ROOT_VAR; END_VAR
    rtti_root_var_is_retain := (v.data^.flags AND RTTI_RETAIN) <> 0;
END_FUNCTION

FUNCTION rtti_root_var_is_persistent : BOOL
VAR_INPUT v: RTTI_ROOT_VAR; END_VAR
    rtti_root_var_is_persistent := (v.data^.flags AND RTTI_PERSISTENT) <> 0;
END_FUNCTION
